<?php

require_once __DIR__ . '/Database.php';

/**
 * AchievementService gerencia conquistas e medalhas do sistema.
 * Permite listar as conquistas cadastradas e obter as conquistas
 * associadas a um usuário.  Em implementações futuras, poderá
 * incluir lógica para conceder conquistas automaticamente.
 */
class AchievementService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Lista todas as conquistas existentes no sistema.  Esta função
     * pode ser usada no painel administrativo ou no app para
     * mostrar conquistas pendentes.
     *
     * @return array
     */
    public function listAll(): array
    {
        $stmt = $this->conn->prepare('SELECT id, title, description, category, intelligence_id, threshold FROM achievements ORDER BY id');
        $stmt->execute();
        return ['success' => true, 'achievements' => $stmt->fetchAll(PDO::FETCH_ASSOC)];
    }

    /**
     * Retorna as conquistas obtidas por um usuário.
     *
     * @param int $userId
     * @return array
     */
    public function getUserAchievements(int $userId): array
    {
        $stmt = $this->conn->prepare('SELECT a.id, a.title, a.description, a.category, a.intelligence_id, ua.earned_at FROM achievements a JOIN user_achievements ua ON a.id = ua.achievement_id WHERE ua.user_id = ? ORDER BY ua.earned_at');
        $stmt->execute([$userId]);
        return ['success' => true, 'achievements' => $stmt->fetchAll(PDO::FETCH_ASSOC)];
    }

    /**
     * Cria uma nova conquista.  Utilizado no painel admin.
     *
     * @param array $data Deve conter title, description, category, threshold e opcional intelligence_id
     * @return array
     */
    public function createAchievement(array $data): array
    {
        $title = trim($data['title'] ?? '');
        $desc  = trim($data['description'] ?? '');
        $cat   = $data['category'] ?? null;
        $threshold = isset($data['threshold']) ? (int)$data['threshold'] : null;
        $intelId  = isset($data['intelligence_id']) ? (int)$data['intelligence_id'] : null;
        if ($title === '' || $cat === null) {
            return ['success' => false, 'error' => 'Campos obrigatórios ausentes'];
        }
        $allowed = ['seed','sprout','tree','special'];
        if (!in_array($cat, $allowed, true)) {
            return ['success' => false, 'error' => 'Categoria inválida'];
        }
        try {
            $stmt = $this->conn->prepare('INSERT INTO achievements (title, description, category, intelligence_id, threshold, created_at) VALUES (?,?,?,?,?,NOW())');
            $stmt->execute([$title, $desc, $cat, $intelId, $threshold]);
            return ['success' => true, 'id' => (int)$this->conn->lastInsertId()];
        } catch (Throwable $e) {
            error_log('[AchievementService::createAchievement] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Erro ao criar conquista'];
        }
    }

    /**
     * Atualiza uma conquista existente.
     *
     * @param int $achievementId
     * @param array $data
     * @return array
     */
    public function updateAchievement(int $achievementId, array $data): array
    {
        $fields = [];
        $params = [];
        if (isset($data['title']) && trim($data['title']) !== '') {
            $fields[] = 'title = ?';
            $params[] = trim($data['title']);
        }
        if (isset($data['description'])) {
            $fields[] = 'description = ?';
            $params[] = trim($data['description']);
        }
        if (isset($data['category']) && in_array($data['category'], ['seed','sprout','tree','special'], true)) {
            $fields[] = 'category = ?';
            $params[] = $data['category'];
        }
        if (isset($data['threshold'])) {
            $fields[] = 'threshold = ?';
            $params[] = $data['threshold'] !== null ? (int)$data['threshold'] : null;
        }
        if (array_key_exists('intelligence_id', $data)) {
            // Permitido valor null ou int
            $fields[] = 'intelligence_id = ?';
            $params[] = $data['intelligence_id'] !== null ? (int)$data['intelligence_id'] : null;
        }
        if (empty($fields)) {
            return ['success' => false, 'error' => 'Nenhuma atualização'];
        }
        $params[] = $achievementId;
        $sql = 'UPDATE achievements SET ' . implode(', ', $fields) . ' WHERE id = ?';
        try {
            $stmt = $this->conn->prepare($sql);
            $stmt->execute($params);
            return ['success' => true];
        } catch (Throwable $e) {
            error_log('[AchievementService::updateAchievement] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Erro ao atualizar conquista'];
        }
    }
}