<?php

require_once __DIR__ . '/Database.php';

/**
 * AdminIntelligenceService provê CRUD para a entidade intelligences
 * pelo painel administrativo.
 */
class AdminIntelligenceService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Lista todas as inteligências cadastradas.
     *
     * @return array
     */
    public function list(): array
    {
        $stmt = $this->conn->query('SELECT id, name_pt, name_es, name_en, description, created_at FROM intelligences ORDER BY id');
        return ['success' => true, 'intelligences' => $stmt->fetchAll(PDO::FETCH_ASSOC)];
    }

    /**
     * Cria uma nova inteligência.
     *
     * @param array $data Deve conter name_pt, name_es, name_en e opcional description
     * @return array
     */
    public function create(array $data): array
    {
        $pt = trim($data['name_pt'] ?? '');
        $es = trim($data['name_es'] ?? '');
        $en = trim($data['name_en'] ?? '');
        $desc = trim($data['description'] ?? '');
        if ($pt === '' || $es === '' || $en === '') {
            return ['success' => false, 'error' => 'Nome em todos os idiomas é obrigatório'];
        }
        try {
            $stmt = $this->conn->prepare('INSERT INTO intelligences (name_pt, name_es, name_en, description, created_at) VALUES (?,?,?,?,NOW())');
            $stmt->execute([$pt, $es, $en, $desc]);
            return ['success' => true, 'id' => (int)$this->conn->lastInsertId()];
        } catch (Throwable $e) {
            error_log('[AdminIntelligenceService::create] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Erro ao criar inteligência'];
        }
    }

    /**
     * Atualiza uma inteligência.
     *
     * @param int $id
     * @param array $data
     * @return array
     */
    public function update(int $id, array $data): array
    {
        $fields = [];
        $params = [];
        if (isset($data['name_pt']) && trim($data['name_pt']) !== '') {
            $fields[] = 'name_pt = ?';
            $params[] = trim($data['name_pt']);
        }
        if (isset($data['name_es']) && trim($data['name_es']) !== '') {
            $fields[] = 'name_es = ?';
            $params[] = trim($data['name_es']);
        }
        if (isset($data['name_en']) && trim($data['name_en']) !== '') {
            $fields[] = 'name_en = ?';
            $params[] = trim($data['name_en']);
        }
        if (array_key_exists('description', $data)) {
            $fields[] = 'description = ?';
            $params[] = trim($data['description']);
        }
        if (empty($fields)) {
            return ['success' => false, 'error' => 'Nenhuma atualização'];
        }
        $params[] = $id;
        $sql = 'UPDATE intelligences SET ' . implode(', ', $fields) . ' WHERE id = ?';
        try {
            $stmt = $this->conn->prepare($sql);
            $stmt->execute($params);
            return ['success' => true];
        } catch (Throwable $e) {
            error_log('[AdminIntelligenceService::update] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Erro ao atualizar inteligência'];
        }
    }
}