<?php
/*
 * AdminUserController
 *
 * Endpoints administrativos para gerenciar usuários.  Apenas
 * administradores podem acessar estas rotas.  As ações suportadas
 * são:
 *   - GET  (action=list)      : lista usuários (pode filtrar por tipo)
 *   - POST (action=update)    : atualiza dados de um usuário
 */

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once __DIR__ . '/AdminUserService.php';
require_once __DIR__ . '/TokenService.php';
require_once __DIR__ . '/Database.php';

try {
    $service = new AdminUserService();
} catch (Throwable $e) {
    error_log('[AdminUserController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}

/**
 * Função para verificar se o usuário é administrador.
 */
function admin_user_is_admin(int $userId): bool
{
    $conn = get_db_connection();
    $stmt = $conn->prepare('SELECT user_type FROM users WHERE id = ?');
    $stmt->execute([$userId]);
    return $stmt->fetchColumn() === 'admin';
}

$method = $_SERVER['REQUEST_METHOD'];
$raw    = file_get_contents('php://input');
$data   = json_decode($raw, true);
if (!is_array($data)) {
    $data = $_POST ?: [];
}

try {
    $uid = require_token();
    if (!admin_user_is_admin($uid)) {
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'Acesso negado']);
        exit;
    }
    if ($method === 'GET') {
        $action = $_GET['action'] ?? 'list';
        switch ($action) {
            case 'list':
                $type = $_GET['type'] ?? null;
                $res = $service->listUsers($type);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Ação inválida']);
                exit;
        }
    } elseif ($method === 'POST') {
        $action = $_GET['action'] ?? $data['action'] ?? null;
        switch ($action) {
            case 'update':
                $targetId = isset($data['id']) ? (int)$data['id'] : 0;
                $res = $service->updateUser($targetId, $data);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Ação inválida']);
                exit;
        }
    }
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Método não suportado']);
    exit;
} catch (Throwable $e) {
    error_log('[AdminUserController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}