<?php

require_once __DIR__ . '/Database.php';

/**
 * AdminUserService fornece operações administrativas para gerenciar
 * usuários.  Permite listar todos os usuários, atualizar dados e
 * (opcionalmente) remover contas.  Todas as chamadas devem ser
 * protegidas pelo controller que verifica se o usuário é admin.
 */
class AdminUserService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Lista usuários com alguns campos básicos.  Aceita filtro de
     * tipo (responsible, teen, admin) opcional via $type.
     *
     * @param string|null $type
     * @return array
     */
    public function listUsers(?string $type = null): array
    {
        $sql = 'SELECT id, name, email, user_type, language, created_at FROM users';
        $params = [];
        if ($type) {
            $sql .= ' WHERE user_type = ?';
            $params[] = $type;
        }
        $sql .= ' ORDER BY created_at DESC';
        $stmt = $this->conn->prepare($sql);
        $stmt->execute($params);
        return ['success' => true, 'users' => $stmt->fetchAll(PDO::FETCH_ASSOC)];
    }

    /**
     * Atualiza dados do usuário (somente campos permitidos).  Não
     * atualiza senha nem token.  Aceita user_type apenas se for
     * 'responsible','teen' ou 'admin'.
     *
     * @param int $userId
     * @param array $data
     * @return array
     */
    public function updateUser(int $userId, array $data): array
    {
        if ($userId <= 0) {
            return ['success' => false, 'error' => 'ID inválido'];
        }
        $fields = [];
        $params = [];
        if (isset($data['name']) && trim($data['name']) !== '') {
            $fields[] = 'name = ?';
            $params[] = trim($data['name']);
        }
        if (isset($data['email']) && filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $fields[] = 'email = ?';
            $params[] = $data['email'];
        }
        if (isset($data['user_type']) && in_array($data['user_type'], ['responsible','teen','admin'], true)) {
            $fields[] = 'user_type = ?';
            $params[] = $data['user_type'];
        }
        if (isset($data['language']) && in_array($data['language'], ['pt','es','en'], true)) {
            $fields[] = 'language = ?';
            $params[] = $data['language'];
        }
        if (isset($data['date_of_birth']) && $data['date_of_birth'] !== '') {
            $fields[] = 'date_of_birth = ?';
            $params[] = $data['date_of_birth'];
        }
        if (empty($fields)) {
            return ['success' => false, 'error' => 'Nenhum campo válido'];
        }
        $params[] = $userId;
        $sql = 'UPDATE users SET ' . implode(', ', $fields) . ' WHERE id = ?';
        try {
            $stmt = $this->conn->prepare($sql);
            $stmt->execute($params);
            return ['success' => true];
        } catch (Throwable $e) {
            error_log('[AdminUserService::updateUser] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Erro ao atualizar usuário'];
        }
    }
}