<?php

require_once __DIR__ . '/Database.php';

/**
 * ChallengeService fornece operações para recuperar desafios diários
 * e registrar respostas de desafios.  Um desafio pode permitir
 * respostas multimídia e a validação (automática ou manual) é
 * processada aqui.
 */
class ChallengeService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Lista os desafios disponíveis para o usuário em uma data.  O
     * parâmetro $date deve ser no formato YYYY‑MM‑DD.  Retorna cada
     * desafio com informações básicas e, se houver, o status de
     * conclusão do usuário.
     *
     * @param int $userId
     * @param string $date
     * @return array
     */
    public function getDailyChallenges(int $userId, string $date): array
    {
        $stmt = $this->conn->prepare('SELECT c.id, c.title, c.description, c.start_time, c.end_time, c.points,
            uc.completed_at IS NOT NULL AS completed
            FROM challenges c
            LEFT JOIN user_challenges uc ON c.id = uc.challenge_id AND uc.user_id = ?
            WHERE (c.start_time IS NULL OR c.start_time <= ?) AND (c.end_time IS NULL OR c.end_time >= ?)');
        $stmt->execute([$userId, $date, $date]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Submete a resposta de um desafio.  Aceita texto e/ou mídia.  A
     * lógica de avaliação é simplificada; a coluna status é marcada
     * como manual_review.  Retorna pontos ganhos.
     *
     * @param int $userId
     * @param int $challengeId
     * @param array $data
     * @param array $files
     * @return array
     */
    public function submitChallenge(int $userId, int $challengeId, array $data, array $files): array
    {
        // Obter desafio para pontos
        $cstmt = $this->conn->prepare('SELECT id, points FROM challenges WHERE id = ? LIMIT 1');
        $cstmt->execute([$challengeId]);
        $challenge = $cstmt->fetch(PDO::FETCH_ASSOC);
        if (!$challenge) {
            http_response_code(404);
            return ['success' => false, 'error' => 'Desafio não encontrado'];
        }
        $text  = trim($data['response_text'] ?? '');
        $audioUrl = null;
        $imageUrl = null;
        if (isset($files['response_audio']) && $files['response_audio']['tmp_name']) {
            $audioUrl = $this->saveUploadedFile($files['response_audio'], 'challenge_audio');
        }
        if (isset($files['response_image']) && $files['response_image']['tmp_name']) {
            $imageUrl = $this->saveUploadedFile($files['response_image'], 'challenge_image');
        }
        $validationStatus = 'manual_review';
        $feedback = null;
        $pointsEarned = (int)$challenge['points'];
        // Insere em challenge_responses
        try {
            $ins = $this->conn->prepare('INSERT INTO challenge_responses (user_id, challenge_id, response_text, response_audio_url, response_image_url, validation_status, validation_feedback, submitted_at, validated_at, points_earned) VALUES (?,?,?,?,?,?,?,?,?,?)');
            $ins->execute([
                $userId,
                $challengeId,
                $text ?: null,
                $audioUrl,
                $imageUrl,
                $validationStatus,
                $feedback,
                date('Y-m-d H:i:s'),
                null,
                $pointsEarned,
            ]);
        } catch (Throwable $e) {
            error_log('[ChallengeService::submitChallenge] erro ao gravar resposta: ' . $e->getMessage());
        }
        // Atualiza user_challenges
        try {
            $sel = $this->conn->prepare('SELECT id FROM user_challenges WHERE user_id = ? AND challenge_id = ? LIMIT 1');
            $sel->execute([$userId, $challengeId]);
            if ($sel->fetch()) {
                $up = $this->conn->prepare('UPDATE user_challenges SET completed_at = NOW(), points_earned = ? WHERE user_id = ? AND challenge_id = ?');
                $up->execute([$pointsEarned, $userId, $challengeId]);
            } else {
                $insUc = $this->conn->prepare('INSERT INTO user_challenges (user_id, challenge_id, started_at, completed_at, points_earned, created_at) VALUES (?,?,?,?,?,NOW())');
                $insUc->execute([$userId, $challengeId, date('Y-m-d H:i:s'), date('Y-m-d H:i:s'), $pointsEarned]);
            }
            // Registrar moedas
            $coins = $this->conn->prepare('INSERT INTO coins_transactions (user_id, amount, transaction_type, source, reference_id, description, created_at) VALUES (?,?,?,?,?,?,NOW())');
            $coins->execute([$userId, $pointsEarned, 'earned', 'challenge', $challengeId, 'Pontuação por desafio']);
        } catch (Throwable $e) {
            error_log('[ChallengeService::submitChallenge] erro ao atualizar user_challenges: ' . $e->getMessage());
        }
        return [
            'success' => true,
            'points'  => $pointsEarned,
            'validation_status' => $validationStatus,
            'feedback' => $feedback,
        ];
    }

    /**
     * Salva arquivo em uploads (mesma implementação de QuizService).  Faz
     * verificação simples e retorna caminho relativo.
     *
     * @param array $file
     * @param string $prefix
     * @return string
     */
    private function saveUploadedFile(array $file, string $prefix): string
    {
        if (!is_dir(__DIR__ . '/../uploads')) {
            mkdir(__DIR__ . '/../uploads', 0777, true);
        }
        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = $prefix . '_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
        $dest = __DIR__ . '/../uploads/' . $filename;
        if (!move_uploaded_file($file['tmp_name'], $dest)) {
            throw new RuntimeException('Falha ao salvar arquivo');
        }
        return 'uploads/' . $filename;
    }
}