<?php

require_once __DIR__ . '/Database.php';

/**
 * DayService lida com consultas e operações relacionadas aos dias de
 * trilhas (cards de conteúdo).  Permite listar os dias de uma trilha,
 * obter detalhes de um dia (incluindo seus conteúdos) e marcar um dia
 * como concluído para um usuário.
 */
class DayService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Lista todos os dias de uma trilha ordenados por day_number.
     *
     * @param int $trailId
     * @return array
     */
    public function getDaysByTrail(int $trailId): array
    {
        $stmt = $this->conn->prepare('SELECT id, day_number, day_type, release_time, title FROM days WHERE trail_id = ? ORDER BY day_number ASC');
        $stmt->execute([$trailId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Obtém detalhes completos de um dia, incluindo os itens de
     * conteúdo.  Retorna null se o dia não existir.
     *
     * @param int $dayId
     * @return array|null
     */
    public function getDay(int $dayId): ?array
    {
        // Dados básicos do dia
        $stmt = $this->conn->prepare('SELECT id, trail_id, day_number, day_type, release_time, title FROM days WHERE id = ? LIMIT 1');
        $stmt->execute([$dayId]);
        $day = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$day) {
            return null;
        }
        // Conteúdos do dia
        $cstmt = $this->conn->prepare('SELECT id, content_type, language, order_index, title, body_text, media_url, reference_id FROM day_contents WHERE day_id = ? ORDER BY order_index ASC');
        $cstmt->execute([$dayId]);
        $contents = $cstmt->fetchAll(PDO::FETCH_ASSOC);
        $day['contents'] = $contents;
        return $day;
    }

    /**
     * Marca o dia como concluído para um usuário.  Registra na
     * tabela user_day_progress e atribui pontos simples.  Retorna
     * número de pontos concedidos.
     *
     * @param int $userId
     * @param int $dayId
     * @return int Pontos atribuídos
     */
    public function completeDay(int $userId, int $dayId): int
    {
        // Verificar se o registro já existe
        $stmt = $this->conn->prepare('SELECT id, status FROM user_day_progress WHERE user_id = ? AND day_id = ? LIMIT 1');
        $stmt->execute([$userId, $dayId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $points = 10; // Pontuação fixa para conclusão; poderá ser calculada com base em complexidade
        if ($row) {
            if ($row['status'] === 'completed') {
                return 0; // já concluído
            }
            // Atualiza registro existente
            $update = $this->conn->prepare('UPDATE user_day_progress SET status = "completed", completed_at = NOW(), points_earned = ? WHERE id = ?');
            $update->execute([$points, $row['id']]);
        } else {
            // Insere novo registro
            $ins = $this->conn->prepare('INSERT INTO user_day_progress (user_id, day_id, status, started_at, completed_at, points_earned, created_at) VALUES (?,?,?,?,?,?,NOW())');
            $ins->execute([$userId, $dayId, 'completed', NULL, date('Y-m-d H:i:s'), $points]);
        }
        // Atualiza progresso da inteligência relacionada
        try {
            $intelligenceId = $this->getIntelligenceIdByDay($dayId);
            if ($intelligenceId) {
                // Calcular progresso: aqui é simplificado; incrementar percentagem
                $sel = $this->conn->prepare('SELECT id, progress_percent, streak_days, last_activity FROM progress_intelligence WHERE user_id = ? AND intelligence_id = ? LIMIT 1');
                $sel->execute([$userId, $intelligenceId]);
                $prow = $sel->fetch(PDO::FETCH_ASSOC);
                if ($prow) {
                    $newPercent = min(100, (float)$prow['progress_percent'] + (100 / 7));
                    $streak = $prow['streak_days'];
                    $last = $prow['last_activity'];
                    $today = date('Y-m-d');
                    if ($last === $today) {
                        // já contou no streak de hoje
                    } elseif ($last === date('Y-m-d', strtotime('-1 day'))) {
                        $streak += 1;
                    } else {
                        $streak = 1;
                    }
                    $up = $this->conn->prepare('UPDATE progress_intelligence SET progress_percent = ?, streak_days = ?, last_activity = ? WHERE id = ?');
                    $up->execute([$newPercent, $streak, $today, $prow['id']]);
                } else {
                    $insP = $this->conn->prepare('INSERT INTO progress_intelligence (user_id, intelligence_id, progress_percent, streak_days, last_activity, created_at) VALUES (?,?,?,?,?,NOW())');
                    $insP->execute([$userId, $intelligenceId, 100 / 7, 1, date('Y-m-d')]);
                }
            }
        } catch (Throwable $e) {
            error_log('[DayService::completeDay] erro ao atualizar progresso: ' . $e->getMessage());
        }
        return $points;
    }

    /**
     * Encontra a inteligência associada a um dia (via trilha).  Retorna
     * null se não encontrar.
     *
     * @param int $dayId
     * @return int|null
     */
    private function getIntelligenceIdByDay(int $dayId): ?int
    {
        $stmt = $this->conn->prepare('SELECT t.intelligence_id FROM days d JOIN trails t ON d.trail_id = t.id WHERE d.id = ? LIMIT 1');
        $stmt->execute([$dayId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? (int)$row['intelligence_id'] : null;
    }
}