<?php
/*
 * ProfileController
 *
 * Controla requisições relacionadas ao perfil do usuário.  As ações
 * são:
 *   - GET (action=get)          : retorna informações completas do perfil
 *   - POST (action=update)      : atualiza dados do perfil
 *   - POST (action=registerDevice) : registra token de notificação
 */

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once __DIR__ . '/ProfileService.php';
require_once __DIR__ . '/TokenService.php';

try {
    $service = new ProfileService();
} catch (Throwable $e) {
    error_log('[ProfileController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$raw    = file_get_contents('php://input');
$data   = json_decode($raw, true);
if (!is_array($data)) {
    $data = $_POST ?: [];
}

try {
    if ($method === 'GET') {
        $userId = require_token();
        $action = $_GET['action'] ?? 'get';
        switch ($action) {
            case 'get':
                $res = $service->getProfile($userId);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Ação inválida']);
                exit;
        }
    } elseif ($method === 'POST') {
        $action = $_GET['action'] ?? $data['action'] ?? null;
        switch ($action) {
            case 'update':
                $userId = require_token();
                $res = $service->updateProfile($userId, $data);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            case 'registerDevice':
                $userId = require_token();
                $token = $data['device_token'] ?? '';
                $platform = $data['platform'] ?? 'android';
                $res = $service->registerDeviceToken($userId, $token, $platform);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Ação inválida']);
                exit;
        }
    }
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Método não suportado']);
    exit;
} catch (Throwable $e) {
    error_log('[ProfileController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}