<?php
/*
 * QuizController
 *
 * Endpoint para quizzes.  Suporta:
 *  - GET com `id` : retorna detalhes de um quiz (pergunta, opções ou pares)
 *  - POST : submete resposta de um usuário para um quiz.  Os campos
 *    aceitos no body são: `quiz_id`, `selected_option_id` (opcional),
 *    `answer_text` (opcional).  Arquivos de áudio/imagem devem ser
 *    enviados em multipart/form‑data como `answer_audio` e
 *    `answer_image`.
 */

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once __DIR__ . '/QuizService.php';
require_once __DIR__ . '/TokenService.php';

try {
    $service = new QuizService();
} catch (Throwable $e) {
    error_log('[QuizController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}

try {
    $method = $_SERVER['REQUEST_METHOD'];
    $userId = require_token();
    if ($method === 'GET') {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'id é obrigatório']);
            exit;
        }
        $quiz = $service->getQuiz($id);
        if (!$quiz) {
            http_response_code(404);
            echo json_encode(['success' => false, 'error' => 'Quiz não encontrado']);
            exit;
        }
        echo json_encode(['success' => true, 'quiz' => $quiz], JSON_UNESCAPED_UNICODE);
        exit;
    } elseif ($method === 'POST') {
        // Para multipart/form, o corpo pode não ser JSON
        $data = [];
        if (isset($_POST['quiz_id'])) {
            $data['quiz_id'] = $_POST['quiz_id'];
            if (isset($_POST['selected_option_id'])) $data['selected_option_id'] = $_POST['selected_option_id'];
            if (isset($_POST['answer_text'])) $data['answer_text'] = $_POST['answer_text'];
        } else {
            // Tentar JSON
            $raw = file_get_contents('php://input');
            $json = json_decode($raw, true);
            if (is_array($json)) {
                $data = $json;
            }
        }
        $quizId = isset($data['quiz_id']) ? (int)$data['quiz_id'] : 0;
        if ($quizId <= 0) {
            http_response_code(422);
            echo json_encode(['success' => false, 'error' => 'quiz_id é obrigatório']);
            exit;
        }
        $result = $service->submitAnswer($userId, $quizId, $data, $_FILES);
        echo json_encode($result, JSON_UNESCAPED_UNICODE);
        exit;
    } else {
        http_response_code(405);
        echo json_encode(['success' => false, 'error' => 'Método não suportado']);
        exit;
    }
} catch (Throwable $t) {
    error_log('[QuizController] ' . $t->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}