<?php

require_once __DIR__ . '/Database.php';

/**
 * RankingService calcula pontuações totais e gera rankings nas
 * categorias geral, escola e família.  Os pontos são somados a
 * partir das tabelas user_day_progress, user_quiz_answers e
 * user_challenges.  Para escola e família, o ranking se restringe
 * aos usuários vinculados ao mesmo contexto do usuário atual.
 */
class RankingService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Obtém ranking geral ou filtrado por escola/família.
     *
     * @param int $userId Usuário solicitante (necessário para determinar família/escola)
     * @param string $scope 'geral' | 'escola' | 'familia'
     * @return array Lista de usuários ordenados por pontos decrescentes
     */
    public function getRanking(int $userId, string $scope = 'geral'): array
    {
        $scope = strtolower($scope);
        $allowed = ['geral','escola','familia'];
        if (!in_array($scope, $allowed, true)) {
            $scope = 'geral';
        }
        // Subquery para somar pontos por usuário
        $pointsSql = 'SELECT user_id, SUM(points) AS total_points FROM (
            SELECT user_id, SUM(points_earned) AS points FROM user_day_progress GROUP BY user_id
            UNION ALL
            SELECT user_id, SUM(points_earned) AS points FROM user_quiz_answers GROUP BY user_id
            UNION ALL
            SELECT user_id, SUM(points_earned) AS points FROM user_challenges GROUP BY user_id
        ) AS union_points GROUP BY user_id';
        $where = '';
        $params = [];
        if ($scope === 'familia') {
            // Obter IDs da família do usuário
            $fStmt = $this->conn->prepare('SELECT family_id FROM user_families WHERE user_id = ?');
            $fStmt->execute([$userId]);
            $familyIds = $fStmt->fetchAll(PDO::FETCH_COLUMN, 0);
            if (!$familyIds) {
                return [];
            }
            // Filtrar usuários que pertencem a essa(s) família(s)
            $placeholders = implode(',', array_fill(0, count($familyIds), '?'));
            $where = "WHERE u.id IN (SELECT user_id FROM user_families WHERE family_id IN ($placeholders))";
            $params = $familyIds;
        } elseif ($scope === 'escola') {
            // Encontrar schools do usuário (via classes)
            $sStmt = $this->conn->prepare('SELECT DISTINCT c.school_id FROM user_classes uc JOIN classes c ON uc.class_id = c.id WHERE uc.user_id = ?');
            $sStmt->execute([$userId]);
            $schoolIds = $sStmt->fetchAll(PDO::FETCH_COLUMN, 0);
            if (!$schoolIds) {
                return [];
            }
            $placeholders = implode(',', array_fill(0, count($schoolIds), '?'));
            $where = "WHERE u.id IN (SELECT uc.user_id FROM user_classes uc JOIN classes c2 ON uc.class_id = c2.id WHERE c2.school_id IN ($placeholders))";
            $params = $schoolIds;
        }
        // Montar consulta
        $sql = "SELECT u.id, u.name, up.total_points FROM users u JOIN ($pointsSql) up ON u.id = up.user_id $where ORDER BY up.total_points DESC, u.name ASC";
        $stmt = $this->conn->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        // Calcular posição
        $rank = 1;
        foreach ($rows as &$r) {
            $r['rank'] = $rank++;
        }
        unset($r);
        return $rows;
    }
}