<?php
/*
 * ReferralController
 *
 * Controlador HTTP para rotas de indicação.  Todas as rotas
 * precisam de autenticação, exceto em alguns cenários em que o
 * usuário submete o código logo após o cadastro (caso deseje
 * proteger a rota, mantenha a exigência de token).  Usa os
 * seguintes parâmetros de ação via GET ou campo JSON:
 *   - action=get   : retorna informações do código de indicação do usuário
 *   - action=submit: o usuário envia um código de indicação para receber bônus
 */

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

// Headers CORS e JSON
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once __DIR__ . '/ReferralService.php';
require_once __DIR__ . '/TokenService.php';

// Instanciar service
try {
    $service = new ReferralService();
} catch (Throwable $e) {
    error_log('[ReferralController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$raw    = file_get_contents('php://input');
$data   = json_decode($raw, true);
if (!is_array($data)) {
    $data = $_POST ?: [];
}

try {
    if ($method === 'GET') {
        // Exige token para obter informações do código
        $userId = require_token();
        $action = $_GET['action'] ?? 'get';
        switch ($action) {
            case 'get':
                $res = $service->getReferralInfo($userId);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Ação inválida']);
                exit;
        }
    } elseif ($method === 'POST') {
        $action = $_GET['action'] ?? $data['action'] ?? null;
        switch ($action) {
            case 'submit':
                // Usuário deve estar logado para submeter código de referência
                $userId = require_token();
                $code = $data['referral_code'] ?? $data['code'] ?? '';
                $res = $service->submitReferral($userId, (string)$code);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Ação inválida']);
                exit;
        }
    }
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Método não suportado']);
    exit;
} catch (Throwable $e) {
    error_log('[ReferralController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}