<?php

require_once __DIR__ . '/Database.php';

/**
 * ReferralService implementa o programa de indicação.  Usuários
 * podem gerar um código único para convidar amigos e receber
 * moedas quando a indicação for aceita.  Novos usuários devem
 * submeter esse código para creditar os bônus.  A tabela
 * `referrals` guarda o vínculo entre o indicador e o indicado.
 */
class ReferralService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Obtém as informações de indicação de um usuário.  Se ele ainda
     * não possuir um código, será gerado um automaticamente.  Também
     * retorna o número de indicações aceitas e moedas obtidas via
     * indicações (somando coins_transactions com source 'referral').
     *
     * @param int $userId
     * @return array
     */
    public function getReferralInfo(int $userId): array
    {
        // Verificar se já existe um código para este usuário
        $stmt = $this->conn->prepare('SELECT id, referral_code FROM referrals WHERE referrer_id = ? AND status = "pending" LIMIT 1');
        $stmt->execute([$userId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            // Criar um novo código pendente
            $row = $this->createReferralCode($userId);
        }
        // Contar quantas indicações foram aceitas
        $countStmt = $this->conn->prepare('SELECT COUNT(*) FROM referrals WHERE referrer_id = ? AND status = "accepted"');
        $countStmt->execute([$userId]);
        $acceptedCount = (int)$countStmt->fetchColumn();
        // Calcular moedas vindas de indicações
        $coinsStmt = $this->conn->prepare('SELECT COALESCE(SUM(amount),0) FROM coins_transactions WHERE user_id = ? AND source = "referral"');
        $coinsStmt->execute([$userId]);
        $coinsEarned = (int)$coinsStmt->fetchColumn();
        return [
            'success' => true,
            'referral_code' => $row['referral_code'],
            'accepted_count' => $acceptedCount,
            'coins_earned' => $coinsEarned,
        ];
    }

    /**
     * Cria e grava um novo código de indicação para o usuário
     * especificado.  Gera uma string alfanumérica de 8 caracteres.
     * Retorna a linha criada (id e referral_code).
     *
     * @param int $userId
     * @return array
     */
    public function createReferralCode(int $userId): array
    {
        // Gera um código curto e garante unicidade
        $code = $this->generateUniqueCode();
        $stmt = $this->conn->prepare('INSERT INTO referrals (referrer_id, referral_code, status, created_at) VALUES (?,?,"pending",NOW())');
        $stmt->execute([$userId, $code]);
        return [
            'id' => (int)$this->conn->lastInsertId(),
            'referral_code' => $code,
        ];
    }

    /**
     * Gera uma string única para o código de indicação.  Utiliza
     * caracteres alfanuméricos e repete até achar um código não
     * utilizado.  Pode lançar uma exceção em caso de falha.
     *
     * @return string
     */
    private function generateUniqueCode(): string
    {
        $attempt = 0;
        do {
            $code = strtoupper(substr(bin2hex(random_bytes(4)), 0, 8));
            $check = $this->conn->prepare('SELECT id FROM referrals WHERE referral_code = ? LIMIT 1');
            $check->execute([$code]);
            $exists = $check->fetch();
            $attempt++;
            if ($attempt > 10) {
                throw new RuntimeException('Falha ao gerar código de referência único');
            }
        } while ($exists);
        return $code;
    }

    /**
     * Submete um código de referência para o usuário atual.  Define o
     * status como aceito na tabela `referrals`, associa o id do
     * referido e registra moedas para ambos.  Caso o código seja
     * inválido ou já utilizado, retorna erro.
     *
     * @param int $userId Id do usuário que está se cadastrando/usando o código
     * @param string $code Código de referência enviado
     * @return array
     */
    public function submitReferral(int $userId, string $code): array
    {
        $code = trim($code);
        if ($code === '') {
            return ['success' => false, 'error' => 'Código inválido'];
        }
        // Encontrar registro pendente do código
        $stmt = $this->conn->prepare('SELECT * FROM referrals WHERE referral_code = ? LIMIT 1');
        $stmt->execute([$code]);
        $ref = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$ref) {
            return ['success' => false, 'error' => 'Código não encontrado'];
        }
        if ($ref['status'] !== 'pending') {
            return ['success' => false, 'error' => 'Código já utilizado ou inválido'];
        }
        if ((int)$ref['referrer_id'] === $userId) {
            return ['success' => false, 'error' => 'Não é possível usar seu próprio código'];
        }
        try {
            $this->conn->beginTransaction();
            // Atualiza referral
            $upd = $this->conn->prepare('UPDATE referrals SET referred_user_id = ?, referred_email = NULL, status = "accepted", accepted_at = NOW() WHERE id = ?');
            $upd->execute([$userId, $ref['id']]);
            // Award coins: 50 para ambos por padrão
            $award = 50;
            $coins = $this->conn->prepare('INSERT INTO coins_transactions (user_id, amount, transaction_type, source, reference_id, description, created_at) VALUES (?,?,?,?,?,?,NOW()), (?,?,?,?,?,?,NOW())');
            // Para referrer e referido
            $coins->execute([
                $ref['referrer_id'], $award, 'earned', 'referral', $ref['id'], 'Indicação aceita',
                $userId, $award, 'earned', 'referral', $ref['id'], 'Bônus por usar código de referência',
            ]);
            $this->conn->commit();
            return ['success' => true, 'message' => 'Indicação registrada com sucesso', 'points_awarded' => $award];
        } catch (Throwable $e) {
            $this->conn->rollBack();
            error_log('[ReferralService::submitReferral] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Erro ao registrar indicação'];
        }
    }
}