<?php
/*
 * AchievementController
 *
 * Proporciona endpoints para listar conquistas disponíveis e as
 * conquistas de um usuário.  Também permite criar e atualizar
 * conquistas no painel administrativo (requer conta admin).
 */

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once __DIR__ . '/AchievementService.php';
require_once __DIR__ . '/TokenService.php';
require_once __DIR__ . '/Database.php';

try {
    $service = new AchievementService();
} catch (Throwable $e) {
    error_log('[AchievementController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$raw    = file_get_contents('php://input');
$data   = json_decode($raw, true);
if (!is_array($data)) {
    $data = $_POST ?: [];
}

/**
 * Verifica se um user_id corresponde a um usuário administrador.  Retorna
 * true se o campo user_type for 'admin'.
 *
 * @param int $userId
 * @return bool
 */
function is_admin(int $userId): bool
{
    $conn = get_db_connection();
    $stmt = $conn->prepare('SELECT user_type FROM users WHERE id = ?');
    $stmt->execute([$userId]);
    $type = $stmt->fetchColumn();
    return $type === 'admin';
}

try {
    if ($method === 'GET') {
        $action = $_GET['action'] ?? 'list';
        switch ($action) {
            case 'list':
                // Lista todas as conquistas (não precisa autenticação)
                $res = $service->listAll();
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            case 'my':
                $userId = require_token();
                $res = $service->getUserAchievements($userId);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Ação inválida']);
                exit;
        }
    } elseif ($method === 'POST') {
        $action = $_GET['action'] ?? $data['action'] ?? null;
        switch ($action) {
            case 'create':
                $uid = require_token();
                if (!is_admin($uid)) {
                    http_response_code(403);
                    echo json_encode(['success' => false, 'error' => 'Acesso negado']);
                    exit;
                }
                $res = $service->createAchievement($data);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            case 'update':
                $uid = require_token();
                if (!is_admin($uid)) {
                    http_response_code(403);
                    echo json_encode(['success' => false, 'error' => 'Acesso negado']);
                    exit;
                }
                $achievementId = isset($data['id']) ? (int)$data['id'] : 0;
                if ($achievementId <= 0) {
                    http_response_code(400);
                    echo json_encode(['success' => false, 'error' => 'ID de conquista inválido']);
                    exit;
                }
                $res = $service->updateAchievement($achievementId, $data);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Ação inválida']);
                exit;
        }
    }
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Método não suportado']);
    exit;
} catch (Throwable $e) {
    error_log('[AchievementController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}