<?php
/*
 * AuthController
 *
 * Ponto de entrada HTTP para operações de autenticação.  Segue o
 * padrão do exemplo: ações são determinadas pelo parâmetro `action` em
 * requisições POST.  Possui as seguintes ações:
 *   - login        : autentica um usuário e retorna um token
 *   - register     : cria um novo usuário
 *   - logout       : revoga o token atual
 *
 * As respostas são retornadas em JSON com chave `success` e
 * eventualmente `user`, `token`, `expires_at`, `error`.
 */

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

// Headers CORS e JSON
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once __DIR__ . '/AuthService.php';
require_once __DIR__ . '/TokenService.php';

try {
    $service = new AuthService();
} catch (Throwable $e) {
    error_log('[AuthController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$raw    = file_get_contents('php://input');
$data   = json_decode($raw, true);
if (!is_array($data)) {
    $data = $_POST ?: [];
}

try {
    if ($method === 'POST') {
        $action = $_GET['action'] ?? $data['action'] ?? null;
        switch ($action) {
            case 'login':
                $result = $service->login($data);
                if ($result['success']) {
                    $tokenSvc = new TokenService();
                    $tok = $tokenSvc->createToken($result['user']['id']);
                    $result['token'] = $tok['token'];
                    $result['expires_at'] = $tok['expires_at'];
                }
                echo json_encode($result, JSON_UNESCAPED_UNICODE);
                exit;
            case 'register':
                $result = $service->register($data);
                echo json_encode($result, JSON_UNESCAPED_UNICODE);
                exit;
            case 'logout':
                $token = get_bearer_token();
                if ($token) {
                    try { (new TokenService())->revokeToken($token); } catch (Throwable $e) { /* ignore */ }
                }
                echo json_encode(['success' => true]);
                exit;
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Ação inválida']);
                exit;
        }
    }
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Método não suportado']);
    exit;
} catch (Throwable $t) {
    error_log('[AuthController] ' . $t->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}