<?php

require_once __DIR__ . '/Database.php';

/**
 * AuthService lida com operações de autenticação e registro de usuários.
 * As senhas são armazenadas usando password_hash() e verificadas
 * com password_verify().  Suporta criação de usuários do tipo
 * responsável (parent), adolescente (teen) e admin.  Também
 * processa código de indicação caso informado no registro.
 */
class AuthService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Autentica um usuário.  Retorna array com success e user se
     * credenciais forem válidas.  O parâmetro $data deve conter
     * 'email' e 'password'.
     *
     * @param array $data
     * @return array
     */
    public function login(array $data): array
    {
        $email = strtolower(trim($data['email'] ?? ''));
        $password = $data['password'] ?? '';
        if ($email === '' || $password === '') {
            http_response_code(422);
            return ['success' => false, 'error' => 'Email e senha são obrigatórios'];
        }
        try {
            $stmt = $this->conn->prepare('SELECT id, name, email, password_hash, user_type, language FROM users WHERE email = ? LIMIT 1');
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$user) {
                http_response_code(401);
                return ['success' => false, 'error' => 'Credenciais inválidas'];
            }
            $hash = $user['password_hash'];
            $valid = false;
            // Caso legacy: se não começar com prefixo bcrypt, comparar em texto puro
            if (strpos($hash, '$2') === 0) {
                $valid = password_verify($password, $hash);
            } else {
                $valid = ($password === $hash);
            }
            if (!$valid) {
                http_response_code(401);
                return ['success' => false, 'error' => 'Credenciais inválidas'];
            }
            $resultUser = [
                'id'       => (int)$user['id'],
                'name'     => $user['name'],
                'email'    => $user['email'],
                'user_type'=> $user['user_type'],
                'language' => $user['language'],
            ];
            return ['success' => true, 'user' => $resultUser];
        } catch (Throwable $e) {
            error_log('[AuthService::login] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Erro interno'];
        }
    }

    /**
     * Registra um novo usuário.  Exige 'name', 'email', 'password' e
     * 'user_type'.  Opcionalmente aceita 'language' (pt|es|en) e
     * 'referral_code'.  Cria o usuário e retorna success.
     *
     * @param array $data
     * @return array
     */
    public function register(array $data): array
    {
        $name      = trim($data['name'] ?? '');
        $email     = strtolower(trim($data['email'] ?? ''));
        $password  = $data['password'] ?? '';
        $userType  = strtolower(trim($data['user_type'] ?? 'teen'));
        $language  = strtolower(trim($data['language'] ?? 'pt'));
        $referral  = trim($data['referral_code'] ?? '');
        if ($name === '' || $email === '' || $password === '' || $userType === '') {
            http_response_code(422);
            return ['success' => false, 'error' => 'Nome, email, senha e tipo de usuário são obrigatórios'];
        }
        $allowedTypes = ['responsible','parent','responsável','mother','father','teen','adolescent','admin'];
        // Normalizar tipos: mapear sinônimos para valores suportados
        $map = [
            'parent'    => 'responsible',
            'responsável'=> 'responsible',
            'mother'    => 'responsible',
            'father'    => 'responsible',
            'adolescent'=> 'teen',
            'teenager'  => 'teen',
        ];
        if (isset($map[$userType])) {
            $userType = $map[$userType];
        }
        if (!in_array($userType, ['responsible','teen','admin'], true)) {
            $userType = 'teen';
        }
        // Normalizar idioma
        if (!in_array($language, ['pt','es','en'], true)) {
            $language = 'pt';
        }
        try {
            // Verificar duplicidade de email
            $stmt = $this->conn->prepare('SELECT id FROM users WHERE email = ? LIMIT 1');
            $stmt->execute([$email]);
            if ($stmt->fetch()) {
                http_response_code(409);
                return ['success' => false, 'error' => 'Email já cadastrado'];
            }
            // Hash da senha
            $hash = password_hash($password, PASSWORD_DEFAULT);
            $insert = $this->conn->prepare('INSERT INTO users (name, email, password_hash, user_type, language, created_at) VALUES (?,?,?,?,?,NOW())');
            $insert->execute([$name, $email, $hash, $userType, $language]);
            $userId = (int)$this->conn->lastInsertId();
            // Se houver código de indicação, localizar e marcar como aceito
            if ($referral !== '') {
                try {
                    $refStmt = $this->conn->prepare('SELECT id, referrer_id, status FROM referrals WHERE referral_code = ? LIMIT 1');
                    $refStmt->execute([$referral]);
                    $refRow = $refStmt->fetch(PDO::FETCH_ASSOC);
                    if ($refRow) {
                        $update = $this->conn->prepare('UPDATE referrals SET referred_user_id = ?, status = "accepted", accepted_at = NOW() WHERE id = ?');
                        $update->execute([$userId, $refRow['id']]);
                        // Crédita moedas ao usuário que indicou
                        $coinsStmt = $this->conn->prepare('INSERT INTO coins_transactions (user_id, amount, transaction_type, source, reference_id, description, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())');
                        $coinsStmt->execute([$refRow['referrer_id'], 50, 'earned', 'referral', $refRow['id'], 'Indicação aceita']);
                    }
                } catch (Throwable $e) {
                    error_log('[AuthService::register] erro ao processar referral: ' . $e->getMessage());
                }
            }
            return ['success' => true];
        } catch (Throwable $e) {
            error_log('[AuthService::register] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Erro interno'];
        }
    }
}