<?php
/*
 * DayController
 *
 * Endpoints para operações sobre dias (cards de conteúdo).  Suporta:
 *  - GET com `trail_id` : listar dias de uma trilha
 *  - GET com `id`       : obter um dia específico
 *  - POST com action=complete e `day_id` : marcar dia como concluído
 */

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once __DIR__ . '/DayService.php';
require_once __DIR__ . '/TokenService.php';

try {
    $service = new DayService();
} catch (Throwable $e) {
    error_log('[DayController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}

try {
    $method = $_SERVER['REQUEST_METHOD'];
    // Autentica
    $userId = require_token();
    if ($method === 'GET') {
        $trailId = isset($_GET['trail_id']) ? (int)$_GET['trail_id'] : 0;
        $id      = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($trailId > 0) {
            $days = $service->getDaysByTrail($trailId);
            echo json_encode(['success' => true, 'days' => $days], JSON_UNESCAPED_UNICODE);
            exit;
        }
        if ($id > 0) {
            $day = $service->getDay($id);
            if (!$day) {
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Dia não encontrado'], JSON_UNESCAPED_UNICODE);
                exit;
            }
            echo json_encode(['success' => true, 'day' => $day], JSON_UNESCAPED_UNICODE);
            exit;
        }
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Parâmetro inválido']);
        exit;
    } elseif ($method === 'POST') {
        $raw = file_get_contents('php://input');
        $data = json_decode($raw, true);
        if (!is_array($data)) {
            $data = $_POST ?: [];
        }
        $action = $_GET['action'] ?? $data['action'] ?? null;
        if ($action === 'complete') {
            $dayId = isset($data['day_id']) ? (int)$data['day_id'] : 0;
            if ($dayId <= 0) {
                http_response_code(422);
                echo json_encode(['success' => false, 'error' => 'day_id é obrigatório']);
                exit;
            }
            $points = $service->completeDay($userId, $dayId);
            echo json_encode(['success' => true, 'points' => $points], JSON_UNESCAPED_UNICODE);
            exit;
        }
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Ação inválida']);
        exit;
    } else {
        http_response_code(405);
        echo json_encode(['success' => false, 'error' => 'Método não suportado']);
        exit;
    }
} catch (Throwable $t) {
    error_log('[DayController] ' . $t->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}