<?php
/*
 * NotificationController
 *
 * Rotas para consultar e gerenciar notificações.  O usuário pode
 * listar suas notificações e marcá‑las como lidas.  Administradores
 * podem criar e enviar notificações para todos, classes, famílias
 * ou usuários específicos.
 */

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once __DIR__ . '/NotificationService.php';
require_once __DIR__ . '/TokenService.php';
require_once __DIR__ . '/Database.php';

try {
    $service = new NotificationService();
} catch (Throwable $e) {
    error_log('[NotificationController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$raw    = file_get_contents('php://input');
$data   = json_decode($raw, true);
if (!is_array($data)) {
    $data = $_POST ?: [];
}

/**
 * Função local para checar se o usuário é admin.
 */
function notif_is_admin(int $userId): bool
{
    $conn = get_db_connection();
    $stmt = $conn->prepare('SELECT user_type FROM users WHERE id = ?');
    $stmt->execute([$userId]);
    return $stmt->fetchColumn() === 'admin';
}

try {
    if ($method === 'GET') {
        // Listar notificações do usuário
        $userId = require_token();
        $res = $service->listForUser($userId);
        echo json_encode($res, JSON_UNESCAPED_UNICODE);
        exit;
    } elseif ($method === 'POST') {
        $action = $_GET['action'] ?? $data['action'] ?? null;
        switch ($action) {
            case 'markRead':
                $userId = require_token();
                $notifId = isset($data['notification_id']) ? (int)$data['notification_id'] : 0;
                $res = $service->markRead($userId, $notifId);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            case 'create':
                $userId = require_token();
                if (!notif_is_admin($userId)) {
                    http_response_code(403);
                    echo json_encode(['success' => false, 'error' => 'Acesso negado']);
                    exit;
                }
                $res = $service->createAndSend($data);
                echo json_encode($res, JSON_UNESCAPED_UNICODE);
                exit;
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Ação inválida']);
                exit;
        }
    }
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Método não suportado']);
    exit;
} catch (Throwable $e) {
    error_log('[NotificationController] ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erro interno']);
    exit;
}