<?php

require_once __DIR__ . '/Database.php';

/**
 * NotificationService cuida do cadastro e distribuição de
 * notificações push e da recuperação das notificações para os
 * usuários.  Neste MVP, o envio real (push) não é implementado;
 * a notificação é registrada e, opcionalmente, associada aos
 * usuários.
 */
class NotificationService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Lista as notificações que foram entregues para o usuário.
     * As notificações são retornadas com título, mensagem, data de
     * envio e data de leitura.
     *
     * @param int $userId
     * @return array
     */
    public function listForUser(int $userId): array
    {
        $stmt = $this->conn->prepare('SELECT n.id, n.title, n.message, un.delivered_at, un.read_at FROM notifications n JOIN user_notifications un ON n.id = un.notification_id WHERE un.user_id = ? ORDER BY un.delivered_at DESC');
        $stmt->execute([$userId]);
        return ['success' => true, 'notifications' => $stmt->fetchAll(PDO::FETCH_ASSOC)];
    }

    /**
     * Marca uma notificação como lida para o usuário.
     *
     * @param int $userId
     * @param int $notificationId
     * @return array
     */
    public function markRead(int $userId, int $notificationId): array
    {
        try {
            $upd = $this->conn->prepare('UPDATE user_notifications SET read_at = NOW() WHERE user_id = ? AND notification_id = ? AND read_at IS NULL');
            $upd->execute([$userId, $notificationId]);
            return ['success' => true];
        } catch (Throwable $e) {
            error_log('[NotificationService::markRead] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Falha ao marcar leitura'];
        }
    }

    /**
     * Cria e distribui uma notificação.  Apenas administradores
     * deveriam chamar este método via painel.  De acordo com o
     * target_type, associa a notificação aos usuários correspondentes.
     * target_type: 'all', 'class', 'family', 'user'
     * target_id: se for class_id, family_id ou user_id conforme o tipo
     *
     * @param array $data
     * @return array
     */
    public function createAndSend(array $data): array
    {
        $title   = trim($data['title'] ?? '');
        $message = trim($data['message'] ?? '');
        $sendTime = $data['send_time'] ?? date('Y-m-d H:i:s');
        $language = $data['language'] ?? 'pt';
        $targetType = $data['target_type'] ?? 'all';
        $targetId   = isset($data['target_id']) ? (int)$data['target_id'] : null;
        $allowedTarget = ['all','class','family','user'];
        if ($title === '' || $message === '' || !in_array($targetType, $allowedTarget, true)) {
            return ['success' => false, 'error' => 'Parâmetros inválidos'];
        }
        try {
            $this->conn->beginTransaction();
            // Inserir notificação
            $ins = $this->conn->prepare('INSERT INTO notifications (title, message, send_time, language, target_type, target_id, created_at) VALUES (?,?,?,?,?,?,NOW())');
            $ins->execute([$title, $message, $sendTime, $language, $targetType, $targetId]);
            $nid = (int)$this->conn->lastInsertId();
            // Distribuir para usuários
            $userIds = [];
            if ($targetType === 'all') {
                $u = $this->conn->query('SELECT id FROM users')->fetchAll(PDO::FETCH_COLUMN, 0);
                $userIds = $u;
            } elseif ($targetType === 'class') {
                if ($targetId === null) throw new InvalidArgumentException('target_id obrigatório para class');
                $stmt = $this->conn->prepare('SELECT user_id FROM user_classes WHERE class_id = ?');
                $stmt->execute([$targetId]);
                $userIds = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);
            } elseif ($targetType === 'family') {
                if ($targetId === null) throw new InvalidArgumentException('target_id obrigatório para family');
                $stmt = $this->conn->prepare('SELECT user_id FROM user_families WHERE family_id = ?');
                $stmt->execute([$targetId]);
                $userIds = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);
            } elseif ($targetType === 'user') {
                if ($targetId === null) throw new InvalidArgumentException('target_id obrigatório para user');
                $userIds = [$targetId];
            }
            // Inserir em user_notifications
            if (!empty($userIds)) {
                $placeholders = [];
                $params = [];
                foreach ($userIds as $uid) {
                    $placeholders[] = '(?,?,NOW(),NULL)';
                    $params[] = $uid;
                    $params[] = $nid;
                }
                $sql = 'INSERT INTO user_notifications (user_id, notification_id, delivered_at, read_at) VALUES ' . implode(',', $placeholders);
                $stmtInsert = $this->conn->prepare($sql);
                $stmtInsert->execute($params);
            }
            $this->conn->commit();
            // Em uma implementação real, aqui dispararíamos push para devices dos usuários
            return ['success' => true, 'notification_id' => $nid, 'delivered_users' => count($userIds)];
        } catch (Throwable $e) {
            $this->conn->rollBack();
            error_log('[NotificationService::createAndSend] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Falha ao criar notificação'];
        }
    }
}