<?php

require_once __DIR__ . '/Database.php';

/**
 * ProfileService oferece operações relacionadas ao perfil do usuário:
 * obtenção de dados pessoais, progresso, conquistas, saldo de
 * moedas, assim como atualização de dados e registro de tokens de
 * dispositivo para notificações.
 */
class ProfileService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Obtém informações completas do perfil de um usuário.  Inclui
     * dados básicos, famílias, turmas e escolas, saldo de moedas,
     * pontos totais, progresso por inteligência e conquistas.
     *
     * @param int $userId
     * @return array
     */
    public function getProfile(int $userId): array
    {
        // Dados básicos
        $userStmt = $this->conn->prepare('SELECT id, name, email, user_type, language, date_of_birth, created_at FROM users WHERE id = ?');
        $userStmt->execute([$userId]);
        $user = $userStmt->fetch(PDO::FETCH_ASSOC);
        if (!$user) {
            return ['success' => false, 'error' => 'Usuário não encontrado'];
        }
        // Famílias
        $famStmt = $this->conn->prepare('SELECT f.id, f.name, uf.role FROM families f JOIN user_families uf ON f.id = uf.family_id WHERE uf.user_id = ?');
        $famStmt->execute([$userId]);
        $families = $famStmt->fetchAll(PDO::FETCH_ASSOC);
        // Turmas e escolas
        $classStmt = $this->conn->prepare('SELECT c.id, c.name AS class_name, s.id AS school_id, s.name AS school_name FROM classes c JOIN schools s ON c.school_id = s.id JOIN user_classes uc ON uc.class_id = c.id WHERE uc.user_id = ?');
        $classStmt->execute([$userId]);
        $classes = $classStmt->fetchAll(PDO::FETCH_ASSOC);
        // Saldo de moedas
        $balStmt = $this->conn->prepare('SELECT COALESCE(SUM(amount),0) FROM coins_transactions WHERE user_id = ?');
        $balStmt->execute([$userId]);
        $coinBalance = (int)$balStmt->fetchColumn();
        // Pontos totais (soma de todas as transações earned)
        $pointsStmt = $this->conn->prepare('SELECT COALESCE(SUM(points_earned),0) FROM (
            SELECT points_earned FROM user_day_progress WHERE user_id = ?
            UNION ALL
            SELECT points_earned FROM user_quiz_answers WHERE user_id = ?
            UNION ALL
            SELECT points_earned FROM user_challenges WHERE user_id = ?
        ) AS all_pts');
        $pointsStmt->execute([$userId, $userId, $userId]);
        $totalPoints = (int)$pointsStmt->fetchColumn();
        // Progresso por inteligência
        $progStmt = $this->conn->prepare('SELECT i.id AS intelligence_id, i.name_pt, pi.progress_percent, pi.streak_days FROM intelligences i LEFT JOIN progress_intelligence pi ON i.id = pi.intelligence_id AND pi.user_id = ? ORDER BY i.id');
        $progStmt->execute([$userId]);
        $progress = $progStmt->fetchAll(PDO::FETCH_ASSOC);
        // Conquistas
        $achStmt = $this->conn->prepare('SELECT a.id, a.title, a.description, a.category, ua.earned_at FROM achievements a JOIN user_achievements ua ON a.id = ua.achievement_id WHERE ua.user_id = ?');
        $achStmt->execute([$userId]);
        $achievements = $achStmt->fetchAll(PDO::FETCH_ASSOC);
        // Retorna objeto completo
        return [
            'success' => true,
            'user' => $user,
            'families' => $families,
            'classes' => $classes,
            'coins' => $coinBalance,
            'total_points' => $totalPoints,
            'progress' => $progress,
            'achievements' => $achievements,
        ];
    }

    /**
     * Atualiza campos do perfil.  Permite atualizar nome, idioma,
     * data de nascimento e senha (quando fornecida).  Não permite
     * alterar o e‑mail ou tipo de usuário através desta rota.
     *
     * @param int $userId
     * @param array $data
     * @return array
     */
    public function updateProfile(int $userId, array $data): array
    {
        $fields = [];
        $params = [];
        if (isset($data['name']) && trim($data['name']) !== '') {
            $fields[] = 'name = ?';
            $params[] = trim($data['name']);
        }
        if (isset($data['language']) && in_array($data['language'], ['pt','es','en'], true)) {
            $fields[] = 'language = ?';
            $params[] = $data['language'];
        }
        if (isset($data['date_of_birth']) && $data['date_of_birth'] !== '') {
            $fields[] = 'date_of_birth = ?';
            $params[] = $data['date_of_birth'];
        }
        // Atualizar senha se fornecida (min 6 caracteres)
        if (isset($data['password']) && strlen($data['password']) >= 6) {
            $fields[] = 'password_hash = ?';
            $params[] = password_hash($data['password'], PASSWORD_DEFAULT);
        }
        if (empty($fields)) {
            return ['success' => false, 'error' => 'Nenhum campo para atualizar'];
        }
        $params[] = $userId;
        $sql = 'UPDATE users SET ' . implode(', ', $fields) . ', updated_at = NOW() WHERE id = ?';
        $stmt = $this->conn->prepare($sql);
        try {
            $stmt->execute($params);
            return ['success' => true];
        } catch (Throwable $e) {
            error_log('[ProfileService::updateProfile] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Falha ao atualizar perfil'];
        }
    }

    /**
     * Registra o token de notificação de um dispositivo.  Se já
     * existir para o mesmo usuário e token, atualiza last_active; se
     * o token pertencer a outro usuário, opcionalmente atualiza o
     * user_id.  O campo platform deve ser 'android', 'ios' ou 'web'.
     *
     * @param int $userId
     * @param string $deviceToken
     * @param string $platform
     * @return array
     */
    public function registerDeviceToken(int $userId, string $deviceToken, string $platform): array
    {
        $platform = strtolower($platform);
        if (!in_array($platform, ['android','ios','web'], true)) {
            return ['success' => false, 'error' => 'Plataforma inválida'];
        }
        try {
            // Verifica se token já existe
            $sel = $this->conn->prepare('SELECT id, user_id FROM device_tokens WHERE device_token = ? LIMIT 1');
            $sel->execute([$deviceToken]);
            $existing = $sel->fetch(PDO::FETCH_ASSOC);
            if ($existing) {
                // Se já pertence ao mesmo usuário, apenas atualiza last_active
                $upd = $this->conn->prepare('UPDATE device_tokens SET user_id = ?, platform = ?, last_active = NOW() WHERE id = ?');
                $upd->execute([$userId, $platform, $existing['id']]);
            } else {
                $ins = $this->conn->prepare('INSERT INTO device_tokens (user_id, device_token, platform, created_at, last_active) VALUES (?,?,?,?,NOW())');
                $ins->execute([$userId, $deviceToken, $platform, date('Y-m-d H:i:s')]);
            }
            return ['success' => true];
        } catch (Throwable $e) {
            error_log('[ProfileService::registerDeviceToken] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Falha ao registrar token'];
        }
    }
}