<?php

require_once __DIR__ . '/Database.php';

/**
 * ProgressService fornece métricas de progresso para as
 * inteligências e mapa de jornada.  Útil para telas de progresso no
 * app.
 */
class ProgressService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Retorna o progresso percentual de cada inteligência para o
     * usuário, juntamente com o streak de dias consecutivos.
     *
     * @param int $userId
     * @return array
     */
    public function getIntelligenceProgress(int $userId): array
    {
        $stmt = $this->conn->prepare('SELECT i.id AS intelligence_id, i.name_pt, i.name_es, i.name_en, COALESCE(pi.progress_percent,0) AS progress_percent, COALESCE(pi.streak_days,0) AS streak_days FROM intelligences i LEFT JOIN progress_intelligence pi ON i.id = pi.intelligence_id AND pi.user_id = ? ORDER BY i.id');
        $stmt->execute([$userId]);
        return ['success' => true, 'progress' => $stmt->fetchAll(PDO::FETCH_ASSOC)];
    }

    /**
     * Calcula o mapa de jornada transformando progressos em níveis
     * (semente, muda, árvore) com base em thresholds simples.
     *
     * @param int $userId
     * @return array
     */
    public function getJourneyMap(int $userId): array
    {
        $progressData = $this->getIntelligenceProgress($userId);
        if (!$progressData['success']) {
            return $progressData;
        }
        $journey = [];
        foreach ($progressData['progress'] as $p) {
            $level = 'seed';
            $percent = floatval($p['progress_percent']);
            if ($percent >= 66.67) {
                $level = 'tree';
            } elseif ($percent >= 33.34) {
                $level = 'sprout';
            }
            $journey[] = [
                'intelligence_id' => $p['intelligence_id'],
                'name_pt' => $p['name_pt'],
                'name_es' => $p['name_es'],
                'name_en' => $p['name_en'],
                'progress_percent' => $percent,
                'streak_days' => (int)$p['streak_days'],
                'level' => $level,
            ];
        }
        return ['success' => true, 'journey' => $journey];
    }
}