<?php

require_once __DIR__ . '/Database.php';

/**
 * QuizService encapsula lógica de consulta de quizzes e submissão de
 * respostas.  Permite recuperar quiz completo (incluindo opções ou
 * pares) e registrar a resposta de um usuário, persistindo mídia
 * quando enviada e avaliando a resposta por meio de regras simples
 * (e, futuramente, integração com ChatGPT).
 */
class QuizService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Retorna quiz completo pelo id.  Inclui lista de opções ou
     * pares dependendo do tipo de quiz.  Retorna null se não
     * encontrado.
     *
     * @param int $quizId
     * @return array|null
     */
    public function getQuiz(int $quizId): ?array
    {
        $stmt = $this->conn->prepare('SELECT id, day_id, quiz_type, question, language, points FROM quizzes WHERE id = ? LIMIT 1');
        $stmt->execute([$quizId]);
        $quiz = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$quiz) {
            return null;
        }
        // Carrega opções ou pares conforme tipo
        if (in_array($quiz['quiz_type'], ['multiple_choice','true_false'], true)) {
            $optStmt = $this->conn->prepare('SELECT id, option_text, is_correct FROM quiz_options WHERE quiz_id = ? ORDER BY id ASC');
            $optStmt->execute([$quizId]);
            $quiz['options'] = $optStmt->fetchAll(PDO::FETCH_ASSOC);
        } elseif ($quiz['quiz_type'] === 'match') {
            $pairStmt = $this->conn->prepare('SELECT id, prompt, `match` FROM quiz_pairs WHERE quiz_id = ? ORDER BY id ASC');
            $pairStmt->execute([$quizId]);
            $quiz['pairs'] = $pairStmt->fetchAll(PDO::FETCH_ASSOC);
        }
        return $quiz;
    }

    /**
     * Submete a resposta de um usuário para um quiz.  Pode incluir
     * texto, id de opção escolhida, áudio e imagem.  Persiste as
     * respostas na tabela `user_quiz_answers` e calcula a
     * pontuação.  Retorna array com `success`, `points` e
     * opcionalmente `is_correct` e `feedback`.
     *
     * @param int $userId
     * @param int $quizId
     * @param array $data Campos enviados no corpo (selected_option_id, answer_text)
     * @param array $files Arquivos enviados (answer_audio, answer_image)
     * @return array
     */
    public function submitAnswer(int $userId, int $quizId, array $data, array $files): array
    {
        // Verificar se já respondeu (pode permitir múltiplas tentativas se desejar)
        // Para MVP, simplesmente inserimos uma nova linha sempre.
        // Carrega quiz para pontuação e validação
        $quiz = $this->getQuiz($quizId);
        if (!$quiz) {
            http_response_code(404);
            return ['success' => false, 'error' => 'Quiz não encontrado'];
        }
        $selectedOptionId = isset($data['selected_option_id']) ? (int)$data['selected_option_id'] : null;
        $answerText = trim($data['answer_text'] ?? '');
        // Lida com arquivos
        $audioUrl = null;
        $imageUrl = null;
        if (isset($files['answer_audio']) && $files['answer_audio']['tmp_name']) {
            $audioUrl = $this->saveUploadedFile($files['answer_audio'], 'audio');
        }
        if (isset($files['answer_image']) && $files['answer_image']['tmp_name']) {
            $imageUrl = $this->saveUploadedFile($files['answer_image'], 'image');
        }
        // Avaliar resposta
        $isCorrect = null;
        $validationStatus = 'pending';
        $validationFeedback = null;
        $pointsEarned = 0;
        try {
            if ($quiz['quiz_type'] === 'multiple_choice' || $quiz['quiz_type'] === 'true_false') {
                if ($selectedOptionId) {
                    $stmt = $this->conn->prepare('SELECT is_correct FROM quiz_options WHERE id = ? AND quiz_id = ? LIMIT 1');
                    $stmt->execute([$selectedOptionId, $quizId]);
                    $opt = $stmt->fetch(PDO::FETCH_ASSOC);
                    if ($opt) {
                        $isCorrect = (bool)$opt['is_correct'];
                        $validationStatus = 'valid';
                    }
                }
            } else {
                // Para perguntas de texto, match, fill_in ou dissertativa, faça avaliação simples ou chame IA
                if ($answerText !== '') {
                    list($isCorrect, $validationStatus, $validationFeedback) = $this->evaluateTextAnswer($quiz, $answerText);
                } elseif ($audioUrl || $imageUrl) {
                    // Avalia áudio/imagem via IA; aqui utilizamos stub
                    list($isCorrect, $validationStatus, $validationFeedback) = $this->evaluateMediaAnswer($quiz, $audioUrl ?: $imageUrl);
                }
            }
            if ($isCorrect === true) {
                $pointsEarned = (int)$quiz['points'];
            }
        } catch (Throwable $e) {
            error_log('[QuizService::submitAnswer] erro ao avaliar: ' . $e->getMessage());
            $validationStatus = 'manual_review';
        }
        // Armazena a resposta
        try {
            $stmt = $this->conn->prepare('INSERT INTO user_quiz_answers (user_id, quiz_id, selected_option_id, answer_text, answer_audio_url, answer_image_url, is_correct, validation_status, validation_feedback, answered_at, points_earned, created_at) VALUES (?,?,?,?,?,?,?,?,?,NOW(),?,NOW())');
            $stmt->execute([
                $userId,
                $quizId,
                $selectedOptionId ?: null,
                $answerText ?: null,
                $audioUrl,
                $imageUrl,
                $isCorrect,
                $validationStatus,
                $validationFeedback,
                $pointsEarned,
            ]);
        } catch (Throwable $e) {
            error_log('[QuizService::submitAnswer] erro ao gravar resposta: ' . $e->getMessage());
        }
        // Atualiza progresso (pontos, achievements etc.) se desejar
        // Aqui simplificamos e apenas adicionamos pontos à conta do usuário
        if ($pointsEarned > 0) {
            try {
                $insCoins = $this->conn->prepare('INSERT INTO coins_transactions (user_id, amount, transaction_type, source, reference_id, description, created_at) VALUES (?,?,?,?,?,?,NOW())');
                $insCoins->execute([$userId, $pointsEarned, 'earned', 'quiz', $quizId, 'Pontuação por quiz']);
            } catch (Throwable $e) {
                error_log('[QuizService::submitAnswer] erro ao registrar pontos: ' . $e->getMessage());
            }
        }
        return [
            'success' => true,
            'points'  => $pointsEarned,
            'is_correct' => $isCorrect,
            'validation_status' => $validationStatus,
            'feedback' => $validationFeedback,
        ];
    }

    /**
     * Salva um arquivo enviado para o diretório uploads e retorna a
     * URL relativa.  Gera um nome único baseado no prefixo, data e
     * id aleatório.  Lança exceção em caso de falha.
     *
     * @param array $file Estrutura do $_FILES
     * @param string $prefix Prefixo no nome do arquivo
     * @return string
     */
    private function saveUploadedFile(array $file, string $prefix): string
    {
        if (!is_dir(__DIR__ . '/../uploads')) {
            mkdir(__DIR__ . '/../uploads', 0777, true);
        }
        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = $prefix . '_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
        $dest = __DIR__ . '/../uploads/' . $filename;
        if (!move_uploaded_file($file['tmp_name'], $dest)) {
            throw new RuntimeException('Falha ao salvar arquivo');
        }
        // Retorna caminho relativo para armazenamento (pode ser adaptado para URL absoluta)
        return 'uploads/' . $filename;
    }

    /**
     * Avalia respostas de texto usando uma regra simples.  Para uma
     * integração real com IA, substitua a implementação por uma
     * chamada à API da OpenAI.  Retorna array [isCorrect,
     * validationStatus, feedback].
     *
     * @param array $quiz
     * @param string $answer
     * @return array
     */
    private function evaluateTextAnswer(array $quiz, string $answer): array
    {
        // Como exemplo, consideramos qualquer resposta de texto válida, mas retornamos manual_review para avaliação
        $feedback = null;
        $status = 'manual_review';
        $isCorrect = null;
        // TODO: integrar com ChatGPT para avaliar resposta
        return [$isCorrect, $status, $feedback];
    }

    /**
     * Avalia respostas de mídia (áudio ou imagem).  Aqui devolvemos
     * manual_review.  Substitua para integração real com IA.
     *
     * @param array $quiz
     * @param string $mediaUrl
     * @return array
     */
    private function evaluateMediaAnswer(array $quiz, string $mediaUrl): array
    {
        $status = 'manual_review';
        $isCorrect = null;
        $feedback = null;
        // TODO: integrar com IA para transcrever/avaliar
        return [$isCorrect, $status, $feedback];
    }
}