<?php

require_once __DIR__ . '/Database.php';

/**
 * RewardService gerencia a listagem e o resgate de recompensas.
 * Os usuários acumulam moedas (coins) por suas ações e podem
 * trocá‑las por prêmios cadastrados.  A tabela `rewards` define
 * cada prêmio e `user_rewards` registra resgates individuais.
 */
class RewardService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Retorna todas as recompensas disponíveis.  Filtra por data de
     * validade e quantidade disponível.  Campos extras de idioma
     * poderiam ser adicionados; aqui retornamos título e descrição
     * padrão.
     *
     * @return array
     */
    public function listRewards(): array
    {
        $today = date('Y-m-d');
        $stmt = $this->conn->prepare('SELECT id, title, description, cost_coins, available_quantity, valid_from, valid_to FROM rewards WHERE (valid_from IS NULL OR valid_from <= ?) AND (valid_to IS NULL OR valid_to >= ?)' );
        $stmt->execute([$today, $today]);
        $rewards = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return ['success' => true, 'rewards' => $rewards];
    }

    /**
     * Resgata uma recompensa para o usuário.  Verifica se há saldo
     * suficiente de moedas, se a recompensa está disponível e se
     * ainda há estoque.  Deduz o custo do saldo e gera um código de
     * voucher.  Retorna código e status.
     *
     * @param int $userId
     * @param int $rewardId
     * @return array
     */
    public function redeemReward(int $userId, int $rewardId): array
    {
        // Buscar recompensa
        $stmt = $this->conn->prepare('SELECT * FROM rewards WHERE id = ? LIMIT 1');
        $stmt->execute([$rewardId]);
        $reward = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$reward) {
            return ['success' => false, 'error' => 'Recompensa não encontrada'];
        }
        // Verificar validade
        $today = date('Y-m-d');
        if (($reward['valid_from'] && $reward['valid_from'] > $today) || ($reward['valid_to'] && $reward['valid_to'] < $today)) {
            return ['success' => false, 'error' => 'Recompensa não disponível'];
        }
        // Checar estoque
        if ($reward['available_quantity'] !== null && (int)$reward['available_quantity'] <= 0) {
            return ['success' => false, 'error' => 'Recompensa indisponível (estoque esgotado)'];
        }
        // Calcular saldo de moedas do usuário
        $balStmt = $this->conn->prepare('SELECT COALESCE(SUM(amount),0) FROM coins_transactions WHERE user_id = ?');
        $balStmt->execute([$userId]);
        $balance = (int)$balStmt->fetchColumn();
        if ($balance < (int)$reward['cost_coins']) {
            return ['success' => false, 'error' => 'Saldo insuficiente'];
        }
        // Gerar voucher e registrar transações
        $voucher = $this->generateVoucherCode();
        try {
            $this->conn->beginTransaction();
            // Inserir user_rewards.  Utiliza NOW() para as datas de resgate e criação
            $insert = $this->conn->prepare('INSERT INTO user_rewards (user_id, reward_id, redeemed_at, voucher_code, status, created_at) VALUES (?, ?, NOW(), ?, "pending", NOW())');
            $insert->execute([$userId, $rewardId, $voucher]);
            $urId = (int)$this->conn->lastInsertId();
            // Deduzir moedas
            $cost = (int)$reward['cost_coins'];
            $trans = $this->conn->prepare('INSERT INTO coins_transactions (user_id, amount, transaction_type, source, reference_id, description, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())');
            $trans->execute([$userId, -$cost, 'spent', 'reward', $rewardId, 'Resgate de recompensa']);
            // Atualizar estoque
            if ($reward['available_quantity'] !== null) {
                $upd = $this->conn->prepare('UPDATE rewards SET available_quantity = available_quantity - 1 WHERE id = ?');
                $upd->execute([$rewardId]);
            }
            $this->conn->commit();
            return ['success' => true, 'voucher_code' => $voucher, 'user_reward_id' => $urId];
        } catch (Throwable $e) {
            $this->conn->rollBack();
            error_log('[RewardService::redeemReward] ' . $e->getMessage());
            return ['success' => false, 'error' => 'Falha ao resgatar recompensa'];
        }
    }

    /**
     * Lista as recompensas já resgatadas pelo usuário com seus
     * respectivos status.
     *
     * @param int $userId
     * @return array
     */
    public function listUserRewards(int $userId): array
    {
        $stmt = $this->conn->prepare('SELECT ur.id, r.title, ur.voucher_code, ur.status, ur.redeemed_at FROM user_rewards ur JOIN rewards r ON ur.reward_id = r.id WHERE ur.user_id = ? ORDER BY ur.redeemed_at DESC');
        $stmt->execute([$userId]);
        return ['success' => true, 'rewards' => $stmt->fetchAll(PDO::FETCH_ASSOC)];
    }

    /**
     * Gera um código aleatório para voucher.  Usa 12 caracteres
     * hexadecimais.
     *
     * @return string
     */
    private function generateVoucherCode(): string
    {
        return strtoupper(substr(bin2hex(random_bytes(6)), 0, 12));
    }
}