<?php

require_once __DIR__ . '/Database.php';

/**
 * TokenService fornece autenticação via token bearer.  Tokens são
 * gerados como strings hexadecimais e armazenados na tabela
 * `user_tokens` com uma data de expiração.  Métodos de apoio
 * permitem validar e revogar tokens.  Use require_token() nos
 * controllers para proteger rotas.
 */
class TokenService
{
    /** @var PDO */
    private $conn;

    public function __construct()
    {
        $this->conn = get_db_connection();
    }

    /**
     * Cria um novo token para o usuário informado.  O tempo de
     * validade padrão é de 7 dias (604800 segundos).  Retorna o
     * token e a data de expiração.
     *
     * @param int $userId
     * @param int $ttlSeconds
     * @return array
     */
    public function createToken(int $userId, int $ttlSeconds = 604800): array
    {
        $token = bin2hex(random_bytes(32));
        $expires = (new DateTimeImmutable('now', new DateTimeZone('UTC')))
            ->modify("+{$ttlSeconds} seconds")
            ->format('Y-m-d H:i:s');
        $stmt = $this->conn->prepare('INSERT INTO user_tokens (user_id, token, expires_at) VALUES (?,?,?)');
        $stmt->execute([$userId, $token, $expires]);
        return ['token' => $token, 'expires_at' => $expires];
    }

    /**
     * Valida um token.  Retorna o id do usuário associado se o token
     * existir e ainda estiver válido; retorna zero em caso de token
     * inexistente ou expirado.
     *
     * @param string $token
     * @return int
     */
    public function validateToken(string $token): int
    {
        $stmt = $this->conn->prepare('SELECT user_id, expires_at FROM user_tokens WHERE token = ? LIMIT 1');
        $stmt->execute([$token]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) return 0;
        if (strtotime($row['expires_at'] . ' UTC') < time()) return 0;
        return (int)$row['user_id'];
    }

    /**
     * Revoga um token (logout).  Remove o registro da tabela.
     *
     * @param string $token
     */
    public function revokeToken(string $token): void
    {
        $stmt = $this->conn->prepare('DELETE FROM user_tokens WHERE token = ?');
        $stmt->execute([$token]);
    }
}

/**
 * Recupera o token bearer do cabeçalho Authorization.  Retorna null
 * caso não exista token válido no cabeçalho.
 *
 * @return string|null
 */
function get_bearer_token(): ?string
{
    $hdr = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['Authorization'] ?? '';
    if (!$hdr && isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
        $hdr = $_SERVER['REDIRECT_HTTP_AUTHORIZATION'];
    }
    if (!$hdr && function_exists('getallheaders')) {
        $headers = getallheaders();
        foreach ($headers as $key => $value) {
            if (strcasecmp($key, 'Authorization') === 0) {
                $hdr = $value;
                break;
            }
        }
    }
    if (!$hdr) return null;
    if (stripos($hdr, 'Bearer ') === 0) {
        return trim(substr($hdr, 7));
    }
    return null;
}

/**
 * Exige um token válido.  Se o token for inválido ou ausente,
     * responde com HTTP 401 e encerra a execução.  Caso contrário,
     * retorna o id do usuário autenticado.
     *
     * @return int
     */
function require_token(): int
{
    $token = get_bearer_token();
    if (!$token) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'AuthRequired']);
        exit;
    }
    $svc = new TokenService();
    $uid = $svc->validateToken($token);
    if ($uid <= 0) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'InvalidToken']);
        exit;
    }
    return $uid;
}